/* This template file is for setting up a difference equation model for
   fitting to data:
   The user supplies routines map(), initst(), initfit(), and can call
   uc(), and uf() for unknown constants and functions.
   
   The example provided is a consumer resource model.
*/

/* Essential include files */
#include <math.h>
#include <stdio.h>
#include "ddefit95.h"
#include "rangen.h"
#include "map.h"       // include for discrete models only

/***************************************************************************/
/* Put your global CONSTANTS next. e.g. double IamAglobalVariable=2.0;     */
/* (Don't be tempted to use global variables - they will almost certainly  */
/* not work in the way you expect, - in particular grad is often called    */
/* with a value of t earlier than some previously used value)              */
/***************************************************************************/
double bc=0.25;
int qar=0; // set to 1 to use Quasi Auto Regression

/***************************************************************************/
/*             Problem specific routines                                   */
/***************************************************************************/



double repro(double x,double m, double a, double b)
// this is a model specific function
{ double r;
  r=pow(x,m);
  r=a*r/(b+r);
  return(r);
}

double survival(double n,double c0,double v)
// this is another model specific function
{ double r;
  r=c0+(1.0-c0)*exp(-n*n/v);
  return(r);
}

void map(s,c,t,swno)
double *s,*c,t;int swno;

/* Routine which takes the state vector forward 1 time unit
*/

{ double a,b,m,n,c0,v,alpha,x;
  int i;
  a=uc(2);b=uc(3);m=uc(4);
  c0=uc(5);v=uc(6);alpha=1;
  if (qar) s[0]=obs(0,s[0],t); // substitute observed for s[0]
  for (i=0;i<3;i++) s[i]=pow((s[i]*bc+1.0),1/bc); // inverse box-cox
  n=s[0]; // population
  x=s[1]; // food
  s[1]=alpha+alpha*survival(n,c0,v)*(1.0+survival(s[2],c0,v)); //food
  s[2]=s[0]; // lagged pop
  s[0]=repro(x,m,a,b)*n;  // pop.
  for (i=0;i<3;i++) if (s[i]<=0.0) s[i]=0.0;
  for (i=0;i<3;i++) s[i]=(pow(s[i],bc)-1.0)/bc; //box-cox
}





void initst(s,c,t) double *s,*c,t;

/* initialise state variables and any global constants here, using c or uc*/


{ int i;
  s[0]=uc(0); // population
  s[1]=uc(1); // food
  s[2]=uc(0); // lagged population
  for (i=0;i<3;i++)  s[i]=(pow(s[i],bc)-1.0)/bc; //box-cox
}

/**************************************************************************/
/* Fit specification routines                                             */
/**************************************************************************/

void initfit(d)
ddefit_control *d;

/* This is where you have to supply details about the model fit, and the graphical
   display of the model fit.
   
   * This routine must not use: uc(),uf(),guf(),iuf() or call any functions which do. 
     If you call these functions in initfit() you will be comitting memory errors.

   * It is a good idea not to edit the comments out of this routine.
   
   * NOTE if you are supplying a floating point number (a double) it must contain a 
     decimal point or e. If you are supplying an integer it must not! C is very unforgiving
     on this point.

   
*/

{ 

/*************************************************************************/
/* Tell program that model is discrete                                   */
/*************************************************************************/
  d->discrete=1;
/***************** The following values must be supplied ******************/
/* The following define numbers of things .......                         */
/**************************************************************************/
  d->no_uc=7;    /* Number of unknown parameters (integer<5000)           */
  d->no_uf=0;    /* Number of unknown functions (integer<=125)            */
  d->no_c=0;     /* Number of (known) model coefficients (integer<5000)   */
  d->no_s=3;     /* Number of state variables (integer<5000)              */
  d->no_fit=1;   /* Number of variables for which there is data to fit    */
  d->t0=0.0;     /* iteration start time                                  */
  d->t1=38.0;    /* Default end time, when simulating                     */
  d->dout=1.0;   /* don't change this !                                   */

/**************************************************************************/
/* Now choose the error model and fitting method.                         */
/* There are 3 fitting methods (all constrained):                         */
/* 0 - Gauss-Newton with steepest descent backup. This is good on small   */
/*     residual problems. It tries both GN and steepest descent direction */
/*     at each step. Can be slow on large residual problems.              */
/* 1 - Quasi-Newton. Usually faster than Gauss-Newton on moderate to      */
/*     large residual problems.                                           */
/* 2 - Iterative least squares/quadratic programming. Very fast on well   */
/*     behaved problems - can diverge otherwise.                          */
/* There are 3 error models:                                              */
/* 0 - Error independent of mean (normal) - single shot fitting - fastest */
/*     and most robust. ONLY this one works when estimating smoothing     */
/*     parameters.                                                        */
/* 1 - Variance proportional to mean (Poisson). Iteratively re-weighted   */
/*     least sqaures objective used. Convergence can be slow (and is not  */
/*     guaranteed - these are not GLMs).                                  */
/* 2 - Standard deviation proportional to mean (constant CV). Same        */
/*     method and comments as model 1.                                    */
/* Finally specify the number of bootstrap restarts to use, this is very  */
/* useful with difficult objectives, but is ignored whenthere are         */
/* smoothing parameters to be estimated.                                  */
/**************************************************************************/

  d->fitmethod=1;
  d->errors=0;
  d->bsr_reps=10;

/**************************************************************************/
/* Now supply starting values and bounds for the unknown parameters...    */
/**************************************************************************/

  d->uco[0]=0.21;  d->cname[0]="N(0)";
  d->uco[1]=1.2;    d->cname[1]="X(0)";
  d->uco[2]=2.99;  d->cname[2]="a";
  d->uco[3]=32.5;   d->cname[3]= "b";
  d->uco[4]=14.3; d->cname[4]=  "m";
  d->uco[5]=0.0015; d->cname[5]= "c0";
  d->uco[6]=2.0;   d->cname[6]= "v";

/**************************************************************************/
/* Supply the type of constraints and any upper and lower bounds on the   */
/* uc's. An unknown coefficient can be of the following types:            */
/* * UNBOUND - no constraints                                             */
/* * B_BELOW - bounded below: lower bound must be supplied in d->uclb[i]  */ 
/* * B_ABOVE - bounded above: upper bound must be supplied in d->uclb[i]  */ 
/* * B_ABOVE|B_BELOW - bounded above and below.                           */
/* * FIXED - will not be changed in fitting.                              */
/**************************************************************************/

  d->uctype[0]=B_BELOW|B_ABOVE;d->uclb[0]= 0.1;d->ucub[0]=2.0;
  d->uctype[1]=B_BELOW|B_ABOVE;d->uclb[1]=0.1;d->ucub[1]=3.0;
  d->uctype[2]=B_BELOW|B_ABOVE;d->uclb[2]=0.0001;d->ucub[2]=100.0;
  d->uctype[3]=B_BELOW|B_ABOVE;d->uclb[3]=0.0001;d->ucub[3]=100.0;
  d->uctype[4]=B_BELOW|B_ABOVE;d->uclb[4]= 0.0001;d->ucub[4]=30.0;
  d->uctype[5]=B_BELOW|B_ABOVE;d->uclb[5]=0.0001;d->ucub[5]=100.0;
  d->uctype[6]=B_BELOW|B_ABOVE;d->uclb[6]=0.0001;d->ucub[6]=100.0;

/**************************************************************************/
/* Definitions for unknown functions.                                     */
/* d->ufdf[i] is maximum  degrees of freedom for ith u.f.                 */
/* d->uft0[i] to d->uft0[i] defines the range of the argument u.f.        */
/* d->uftype[i] can be UNBOUND, INCREASING, DECREASING, B_BELOW, B_ABOVE  */
/*		          RE1, ORIGIN or FIXED or combinations of these using |.  */
/* d->ufub[i] and d->uflb[i] give the optional upper an lower bounds.     */
/* d->ufsp[i] is the smoothing parameter. If any of these are -ve then    */
/*            automatic selection will be used.                           */
/**************************************************************************/

/**************************************************************************/
/* This part deals with the equivalence of state variables and columns    */
/* of data in the input file.                                             */
/* index[i] is the state variable corresponding to input file column i    */
/* where it is assumed that column 0 of the input file is time.           */
/**************************************************************************/

  d->index[1]=0;
  d->dfile="cr.dat"; /* Data file name */
  d->wfile="cr.w";   /* Weight file name */
  

/**************************************************************************/
/* It is possible to specify extra statistics of the input series that    */
/* should be appended to the data vector as "extra" data to be fitted.    */
/* The weight to be given to these statistics can also be specified.      */
/* Available options are:                                                 */
/* NO_STATS   - nothing (option to use if formal statistical stuff is to  */
/*              be done)                                                  */
/* STDEV      - standard deviation of series.                             */
/* ABOLUTEDEV - Mean absulte deviation of data from series mean.          */
/* MEANGRAD   - Mean gradient between data points.                        */
/* MEANFREQ   - Mean frequency of series.                                 */
/* ACF        - Autocorrelation function to first 10 lags.                */
/* These can be combined, e.g. d->statistics=STDEV|ACF|MEANFREQ.          */
/* In fact only ACF is really useful, in my experience. It is useful for  */
/* finding parameters in the right ballpark for long cyclic series.       */
/* Weights to give to statistics are specified in d->wstats. e.g. for the */
/* above example:                                                         */
/* d->wstats[STDEV]=100.0;d->wstats[MEANFREQ]=1.0;d->wstats[ACF]=30.0;    */
/* would specify weights 100, 1 and 30 for the selected statistics.       */ 
/**************************************************************************/ 

  d->statistics=ACF;
  d->wstats[ACF]=0.001;

/**************************************************************************/
/* Provide information on what the output should look like.               */
/**************************************************************************/

  d->no_windows=1;     /* Number of output windows */
  d->lines[0]=2;       /* Number of lines in each window */
  d->wname[0]="Food Quantity"; /* A name for the window */
  d->range[0].y0= -4.0;d->range[0].y1=6.0;  /* Initial y axis range */

/**************************************************************************/
/* d->windex[i] contains the window number and curve number within that   */
/* window which correspond to variable i. Only specify for variables that */
/* you want ouptut. d->label[i] is the name to use for state variable i.  */
/**************************************************************************/

  d->windex[0].win=0;d->windex[0].cur=0;
  d->windex[1].win=0;d->windex[1].cur=1;
  d->label[0]="Population";
  d->label[1]="Quantity";

}

double inuf(int i,double t)

/* This is where unknown function intial values must be set. 
   
   * i is the index of the unknown function and t is its argument. 
   
   * Given this information you must return the value of the ith unknown function at t from
     this function.  
*/

{}

void inboot(bsctype *bsc)

/* This is the routine for setting up bootstrapping, (not bootstrap 
   restarting)
   
   * You can leave it blank, or set bsc->reps=0, if you don't want to 
     bootstrap.
   
   * bsc->parametric: set to 0 for non-parametric, 1 for parametric or 2 
     for semi-parametric.
   
   * bsc->lumped: set to 0 to treat each fitted series separately for 
     bootstrapping purposes, or to 1 to lump the stages in bootstrapping.

   * bsc->iocontrol: 1 to write replicate data to a file, 2 to read 
     replicate data from a file 0 for no i/o of data.

   * bsc->fdname: name of the file to which replicate data (actually weights
     for no-para) will be written. 

   * bsc->fpname: name of file to which replicate parameter estimates will 
     be written.

   * bsc->SSname: (weighted) sum of squares is written to this file, for 
     each rep.

   * bsc->reps: number of bootstrap replicates to generate and fit.

   * bsc->nextra: number of extra data series (see documentation - usually 
     set to zero or leave out)

   * bsc->ignore: user initialised array of fit variables to ignore (see 
     documentation)

   * bsc->restarts[0/1/2]: Number of bootstrap restarts per fit on first 
     second and subsequent reps.

   * bsc->carry_p: set to 1 to carry the best fit parameter vector from the 
     last rep on as the starting parameter vector for the next fit. Set to 0
     to use best fit parameters to original data as starting parameters for 
     each fit. Set to -1 to always use the original starting parameters.

   * bsc->n_start_files: you can read start parameters from a file: this 
     specifies the number of start files to use. 
   
   * bsc->start_file[]: array for start file names - you must initialise 
     this, e.g. for a list of 3 files: 
       bsc->start_file=(char **)calloc(3,sizeof(char*));
     A * in front of the the first filename indicates that this is a 
     parameter file generated by a previous set of bootstrap runs, with a 
     different set of parameters for each b.s. rep. 
       e.g. bsc->start_file[0]="*zd0.bsp"; tells the program that "zd0.bsp" 
     is a text file with one set of starting parameters on each line.

   For more information please see the written documentation. 
*/

{ bsc->parametric=2;
  bsc->lumped=0;
  bsc->iocontrol=1;
  bsc->fdname="para.bsy";
  bsc->fpname="para.bsp";
  bsc->SSname="para.SS";
  bsc->reps=0;
  bsc->nextra=0;
  bsc->ignore=(int *)calloc(4,sizeof(int));
  bsc->restarts[0]=15;
  bsc->restarts[1]=15;
  bsc->restarts[2]=15;
  bsc->carry_p=0;
}

